function lio=markkmedoid(objnum,kmidx,fov,image,oldx,colordx,colormat,showl,fign,im2front);
%
% Usage: lio=markkmedoid(objnum,kmidx,fov,image,oldx,colordx,colormat,showl,fign,im2front);
%
% Given a (i) kmedoid cluster assignment, (ii) image,  (iii) outline 
% index image, (iv) cluster color assignment matrix and (v) color matrix,
% this script will display the image with overlays of each object, rendered
% in the color that was assigned. 
%
% objnum --- N x 1 vector of cell index numbers
% kmidx --- N x 1 vector of k-medoid cluster index assignments
% fov --- the field of view number to use for the cell outlines
% image --- the unerlying image (e.g. transmitted light) to overlay on
%
% oldx --- the image from CellProfiler with the outline information. The
% value of each pixel of an outline is equal to the index of the cell to
% which it belongs.
%
% colordx --- an Ncluster x 1 vector specifying the color assigned to each 
% k-medoid cluster. For cluster j, colordx(j) is the row in the colormat 
% matrix with the RGB values to be used.
% 
% colormat --- an Ncolor x 3 matrix of colors. Each row is a different 
% color. The three columns are the RGB values for the color on the 0-254 
% scale.
%
% The variable "showl" is logical for show label.
% fign is the desired figure panel number
%
% im2front --- if >0, the script puts image pixels>im2front on top of the
% cell outlines so that the underlying image can be better seen.
%
% TRACEPIX is the pixel width to use when drawing the cell outlines.
%
% BLOCKSIZE is the interval over which the cell numbers from each field of
% view are spaced (an integer multiple of BLOCKSIZE is added to the cell
% indces of each field of view, so as to distinguish which field each cell
% came from).
%
% DARKENFACT --- scalar between 0-1 used to darken the background image.
% fontsz --- the font size for the cell labels
%
TRACEPIX=1;
BLOCKSIZE=1000;
DARKENFACT=0.6;
fontsz=10;
%
% only keep cell objects that belong to the specified fov
%
fovobjdx=find(objnum > fov*BLOCKSIZE & objnum < (fov+1)*BLOCKSIZE);
objnum=objnum(fovobjdx)-fov*BLOCKSIZE;
kmidx=kmidx(fovobjdx);
nobj=size(objnum,1);
%
% create an Ncluster x 3 matrix specifying RGB values
%
objcoldx=colordx(kmidx); objcol=colormat(objcoldx,:);
%
% open figure panel
%
figure(fign);

% %
% % *******************************
% %
% % OPTION 1:
% % The fastest way to draw the outlines, but also shows the cell objects 
% % that were rejected by CellProfiler because of unreasonable nucleus size 
% % and/or edge contact. The rejected cells will be outlined in black.
% %
% % construct a color lookup table indexed on the object number
% mxobjnum=max(max(oldx));
% objcollookup=zeros(mxobjnum+1,3);
% for dx=1:nobj
%     objcollookup(objnum(dx)+1,:)=objcol(dx,:);
% end
% objcollookup = uint8(objcollookup);
% %
% % construct an rgb image of just the colored outlines
% outlines=zeros(size(image));
% oldx=oldx+1;
% %
% foo=objcollookup(:,1); outlines(:,:,1)=foo(oldx);
% foo=objcollookup(:,2); outlines(:,:,2)=foo(oldx);
% foo=objcollookup(:,3); outlines(:,:,3)=foo(oldx);
% %
% % construct and display a composite image of backround and outlines 
% image=DARKENFACT*image;
% oldx=repmat(oldx,[1 1 3]); drawdx=find(oldx>1);
% image(drawdx)=outlines(drawdx);
% imshow(image);

% %
% % *******************************
% %
% % OPTION 2:
% % like OPTION 1 above, sets outline pixels to the color of the cluster to
% % which each cell was assigned, but does not show rejected cells. Slower
% % than OPION 1.
% %
% % for each cell, find the outline pixels and recolor them according to 
% % the cluster to which the cell was assigned.
% image=DARKENFACT*image;
% for dx=1:nobj
%     celldx=objnum(dx);
%     % find pixels for outline of specific cell and color them
%     [olpix1,olpix2]=find(oldx==celldx);
%     for pixdx=1:length(olpix1)
%         image(olpix1(pixdx),olpix2(pixdx),:)=objcol(dx,:);
%     end
% end
% %
% % display the composite image of backround and outlines
% imshow(image);

%
% *******************************
%
% OPTION 3:
% using the plot function, mark each CellProfiler outline pixel with a
% symbol colored according to cell type. Overlay the plot onto the 
% original image. This works well for examing images inside Matlab,
% because the plot symbols are dynamically updated. Also, it is possible to
% add cell index lables. It is not so great when saving an image, because
% the symbols at low zoom are very thick and obscure the underlying data.
%
% concert the 0-255 scale RGB colors to 0-1 scale. Display the background
% image and put a hold on the panel to can plot the outlines.
objcol=objcol/255;
imshow(DARKENFACT*image);
hold on
%
% for each cell, find the outline pixels. Then create a plot of the 
% outline in the color of the cluster to which the cell was assigned.
% Optionally add a label indicating the cell's index number.
for dx=1:nobj
    celldx=objnum(dx);
    % find pixels for outline of specific cell and make a plot
    [olpix1,olpix2]=find(oldx==celldx);
    plot(olpix2, olpix1,'.', 'Color', objcol(dx,:), 'LineWidth', TRACEPIX);
    %
    if (showl) % add the cell label
        xwr=round(median(olpix1));
        ywr=round(median(olpix2));
        text(ywr+fontsz,xwr,num2str(celldx),'FontSize',fontsz,'Color','red');
    end
end
%
% move background image pixels >im2front intensity into foreground
if im2front>0
    alpha=logical(image>=im2front);
    h=imshow(image);
    set(h,'AlphaData',alpha);
end
hold off

% %
% % *******************************
% %
lio=1;
